/***************************************************************
 *                   Materials Object Library                  *
 *              Class isotropic_elasticity_tensor              *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2002,2003,2004,2005,2006,2008 COLLARD Christophe
 * copyright © 2002,2003,2004,2005,2006,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::isotropic_elasticity_tensor
    \brief \f$ 4^{th} \f$ order isotropic elasticity tensor computation

    \htmlonly 
    <FONT color="#838383">

    isotropic elasticity tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let's note \f$ \lambda \f$ and \f$ \mu \f$ the Lame constants. The \f$ 4^{th} \f$ order isotropic elasticity tensor then reads \f$ C_{ijkl} = \lambda \delta_{ij} \delta_{kl} + \mu (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$. \n

    Convertion table : \n
    \f$ \newcommand{\D}{\displaystyle}
    \text{\begin{tabular}{|c|c|c|c|c|c|}
    \hline 
    & Lame modulus & Shear modulus & Young's modulus & Poisson's ration & Bulk modulus \\
    & $\lambda$ & $\mu$ & $E$ & $\nu$ & $K$ \\
    \hline
    & & & & & \\
    $\lambda, \mu$ & & & $\D \frac{\mu(3\lambda + 2\mu)}{\lambda+\mu}$ & $\D \frac{\lambda}{2(\lambda + \mu)}$ & $\D \frac{3\lambda + 2\mu}{3}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, E$ & & irrational & & irrational & irrational \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, \nu$ & & $\D \frac{\lambda(1-2\nu)}{2\nu}$ & $\D \frac{\lambda(1+\nu)(1-2\nu)}{\nu}$ & & $\D \frac{\lambda(1+\nu)}{3\nu}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, K$ & & $\D \frac{3(K-\lambda)}{2}$ & $\D \frac{9K(K-\lambda)}{3K-\lambda}$ & $\D \frac{\lambda}{3K-\lambda}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu. E$ & $\D \frac{\mu(2\mu-E)}{E-3\mu}$ & & & $\D \frac{E-2\mu}{2\mu}$ & $\D \frac{\mu E}{3(3\mu-E)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu, \nu$ & $\D \frac{2\mu\nu}{1-2\nu}$ & & $2\mu(1+\nu)$ & & $\D \frac{2\mu(1+\nu)}{3(1-2\nu)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu, K$ & $\D \frac{3K-2\mu}{3}$ & & $\D \frac{9K\mu}{3K+\mu}$ & $\D \frac{3K-2\mu}{2(3K+\mu)}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $E, \nu$ & $\D \frac{\nu E}{(1+\nu)(1-2\nu)}$ & $\D \frac{E}{2(1+\nu)}$ & & & $\D \frac{E}{3(1-2\nu)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $E, K$ & $\D \frac{3K(3K-E)}{9K-E}$ & $\D \frac{3EK}{9K-E}$ & & $\D \frac{3K-E}{6K}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $\nu, K$ & $\D \frac{3K\nu}{1+\nu}$ & $\D \frac{3K(1-2\nu)}{2(1+\nu)}$ & $3K(1-2\nu)$ & & \\
    & & & & & \\
    \hline
    \end{tabular}} \f$

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2008 Christophe COLLARD \n
            copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 2002-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type isotropic elasticity tensor
#endif

#ifndef _isotropic_elasticity_tensors_h
#define _isotropic_elasticity_tensors_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__TENSORS4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__ISOTROPIC_TENSORS_H)
#include "MateriOL++/isotropic tensors.h"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//===============================================================================
template <class T> class isotropic_elasticity_tensor : public isotropic_tensor<T>
//===============================================================================
{
  using isotropic_tensor<T>::Lambda;
  using isotropic_tensor<T>::Mu;
  using isotropic_tensor<T>::tensor_name;

  protected :
    T lame, shear, nu, E, K;

  public :
    isotropic_elasticity_tensor (char* name= "");  // default constructor
    isotropic_elasticity_tensor (const tensor4<T>&); // cast conversion
    isotropic_elasticity_tensor (const isotropic_tensor<T>&); // cast convertion
    ~isotropic_elasticity_tensor () {}  // destructor

    // methods to compute the 4th order elasticity tensor
    T Lame    (T);  // returns Lame first constant and computes the isotropic tensor
    T Shear   (T);  // returns Lame second constant and computes the isotropic tensor
    T Young   (T);  // Young's modulus and computes the isotropic tensor
    T Poisson (T);  // Poisson's ratio and computes the isotropic tensor
    T Bulk    (T);  // Bulk modulus and computes the isotropic tensor

    // returns elastic constants
    T Lame    () const;  // returns Lame constant
    T Shear   () const;  // returns Shear modulus
    T Young   () const;  // returns Young's modulus
    T Poisson () const;  // returns Poisson's ratio
    T Bulk    () const;  // returns Bulk modulus

    T lambda (T);
    T mu     (T);
    template <class Tf> friend ostream& operator << (ostream&, const isotropic_elasticity_tensor<Tf>&);
};


//=====Private methods for isotropic elasticity tensor==================================


//=====Public methods for isotropic elasticity tensor===================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order isotropic elasticity tensor.

  \param name tensor name
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> isotropic_elasticity_tensor<T>::isotropic_elasticity_tensor (char* name) : isotropic_tensor<T>(name)
//---------------------------------------------------------------------------------------------------------------------
{
  lame = shear = nu = E = K = 0;
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into an isotropic elastic tensor.

  \param tensor \f$ \displaystyle 4^{th} \f$ order tensor
*/

//-------------------------------------------------------------------------------------------------------
template <class T> isotropic_elasticity_tensor<T>::isotropic_elasticity_tensor (const tensor4<T>& tensor)
//-------------------------------------------------------------------------------------------------------
{
  Lambda = lame = tensor(1,1,2,2);
  Mu = shear = tensor(1,2,1,2);
  assert ((*this) == tensor);
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order isotropic tensor into an isotropic elastic tensor.

  \param tensor \f$ \displaystyle 4^{th} \f$ order isotropic tensor
*/

//----------------------------------------------------------------------------------------------------------------
template <class T> isotropic_elasticity_tensor<T>::isotropic_elasticity_tensor (const isotropic_tensor<T>& tensor)
//----------------------------------------------------------------------------------------------------------------
{
  lame = shear = nu = E = K = 0;
  Lame(tensor.lambda());
  Shear(tensor.mu());
}


/*!
  \brief Computes and returns \f$ \lambda \f$ value (see detailed description).
  \param scalar Lame modulus \f$ \lambda \f$
  \return Lame modulus \f$ \lambda \f$
*/

//------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Lame (T scalar)
//------------------------------------------------------------------
{
  Lambda = lame = scalar;

  if (shear) assert (!nu && !E && !K);

  else if (nu)
    { assert (!shear && !E && !K);
      Mu = .5 * lame * (1./nu-2);
    }

  else if (K)
    { assert (!shear && !nu && !E);
      Mu = 1.5 * (K - lame);
    }

  else assert (!E);

  return lame;
}


/*!
  \brief Computes and returns \f$ \mu \f$ value (see detailed description).
  \param scalar shear modulus \f$ \mu \f$
  \return shear modulus \f$ \mu \f$
*/

//-------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Shear (T scalar)
//-------------------------------------------------------------------
{
  Mu = shear = scalar;

  if (lame) assert (!nu && !E && !K);

  else if (nu)
    { assert (!lame && !E && !K);
      if (abs(nu-.5) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Lambda = shear * nu / (.5 - nu);
    }

  else if (E)
    { assert (!lame && !nu && !K);
      if (abs(E-3*shear) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Lambda = shear * (2*shear - E) / (E - 3*shear);
    }

  else if (K)
    { assert (!lame && !nu && !E);
      Lambda = K - 2/3. * shear;
    }

  return shear;
}


/*!
  \brief Computes and returns \f$ E \f$ value (see detailed description).
  \param scalar Young's modulus \f$ E \f$
  \return Young's modulus \f$ E \f$
*/

//------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Young(T scalar)
//------------------------------------------------------------------
{
  E = scalar;

  if (shear)
    { assert (!lame && !nu && !K);
      if (abs(E-3*shear) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Lambda = shear * (2*shear - E) / (E - 3*shear);
    }

  else if (nu)
    { assert (!lame && !shear && !K);
      if (abs(nu-.5) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Lambda = nu * E / ( (1 + nu) * (1 - 2*nu) );
      Mu = .5 * E / (1 + nu);
    }

  else if (K)
    { assert (!lame && !shear && !nu);
      Lambda = 3 * K * (3*K - E) / (9*K - E);
      Mu = 3 * K * E / (9*K - E);
    }

  else assert (!lame);

  return E;
}


/*!
  \brief Computes and returns \f$ \nu \f$ value (see detailed description).
  \param scalar Poisson's ration \f$ \nu \f$
  \return Poisson's ratio \f$ \nu \f$
*/

//--------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Poisson(T scalar)
//--------------------------------------------------------------------
{
  nu = scalar;
  if (abs(nu-.5) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}

  if (lame)
    { assert (!shear && !E && !K);
      Mu = lame * (.5/nu - 1);
    }

  else if (shear)
    { assert (!lame && !E && !K);
      Lambda = shear * nu / (.5 - nu);
    }

  else if (E)
    { assert (!lame && !shear && !K);
      Lambda = nu * E / ((1+nu) * (1 - 2*nu));
      Mu = .5 * E / (1+nu);
    }

  else if (K)
    { assert (!lame && !shear && !E);
      Lambda = 3 * K * nu / (1+nu);
      Mu = 1.5 * K * (1 - 2*nu) / (1+nu);
    }

  return nu;
}


/*!
  \brief Computes and returns \f$ K \f$ value (see detailed description).
  \param scalar bulk modulus \f$ K \f$
  \return bulk modulus \f$ K \f$
*/

//-----------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Bulk(T scalar)
//-----------------------------------------------------------------
{
  K = scalar;

  if (lame)
    { assert (!shear && !nu && !E);
      if (abs(K-lame) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Mu = 1.5 * (K - lame);
    }

  else if (shear)
    { assert (!lame && !nu && !E);
      Lambda = K - 2*shear /3.;
    }

  else if (nu)
    { assert (!lame && !shear && !E);
      if (abs(nu-.5) < epsilon) {cout << "Use incompressible isotropic tensor library \n"; exit(1);}
      Lambda = 3 * K * nu / (1+nu);
      Mu = 1.5 * K * (1 - 2*nu) / (1+nu);
    }

  else if (E)
    { assert (!lame && !shear && !nu);
      Lambda = 3 * K * (3*K - E) / (9*K - E);
      Mu = 3 * K * E / (9*K - E);
    }

  return K;
}


/*!
  \brief Changes and returns \f$ \lambda \f$ value (see detailed description).
  \param scalar Lame constant \f$ \lambda \f$
  \return \f$ \lambda \f$
*/

//--------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::lambda (T scalar)
//--------------------------------------------------------------------
{
  Lame(scalar);

  return lame;
}


/*!
  \brief Changes and/or returns \f$ \mu \f$ value (see detailed description).
  \param scalar shear modulus \f$ \mu \f$
  \return \f$ \mu \f$
*/

//----------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::mu (T scalar)
//----------------------------------------------------------------
{
  Shear(scalar);

  return shear;
}


//---------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Lame() const
//---------------------------------------------------------------
{
  assert (Lambda && Mu);

  return Lambda;
}


//----------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Shear() const
//----------------------------------------------------------------
{
  assert (Lambda && Mu);

  return Mu;
}


//----------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Young() const
//----------------------------------------------------------------
{
  assert (Lambda && Mu);

  if (E) return E;
  else return Mu * (3*Lambda + 2*Mu) / (Lambda + Mu);
}


//------------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Poisson() const
//------------------------------------------------------------------
{
  assert (Lambda && Mu);

  if (nu) return nu;
  else return .5 * Lambda / (Lambda + Mu);
}


//---------------------------------------------------------------
template <class T> T isotropic_elasticity_tensor<T>::Bulk() const
//---------------------------------------------------------------
{
  assert (Lambda && Mu);

  if (K) return K;
  else return Lambda + 2 * Mu / 3.;
}


//-----------------------------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const isotropic_elasticity_tensor<Tf>& tsr)
//-----------------------------------------------------------------------------------------------
{
  assert (tsr.Lambda && tsr.Mu);

  s << tsr.tensor_name << endl;
  s << "Lame constant : " << tsr.Lambda << endl;
  s << "Shear constant : " << tsr.Mu << endl;
  s << "Young's modulus : " << tsr.Young() << endl;
  s << "Poisson's ratio : " << tsr.Poisson() << endl;
  s << "Bulk modulus : " << tsr.Bulk() << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}


}


#endif
